/*---------------------------------------------------------------------------*
 * TorigoyaMZ_LoadCommand.js v.1.0.1
 *---------------------------------------------------------------------------*
 * 2025/01/21
 *---------------------------------------------------------------------------*
 * Based on Ruたん's SaveCommand2 plugin
 *---------------------------------------------------------------------------*/

/*:
 * @target MZ
 * @plugindesc ロードコマンド追加プラグイン (v.1.0.1)
 * @author [Your Name]
 * @license public domain
 * @version 1.0.1
 *
 * @help
 * ロードコマンド追加プラグイン (v.1.0.1)
 *
 * イベントコマンドの「プラグインコマンド」を使って、
 * イベント中に自動的にロードを実行できるようになります。
 *
 * @command loadWithId
 * @text 【ロード】スロットIDを指定してロード
 * @desc 指定のスロットIDからロードを実行します。
 *
 * @arg slotId
 * @text スロットID
 * @desc ロードするスロットのID
 * @type number
 * @min 0
 * @default 1
 *
 * @command loadWithVariable
 * @text 【ロード】変数でスロットIDを指定してロード
 * @desc 指定した変数の値のスロットIDからロードを実行します
 *
 * @arg slotVariableId
 * @text スロットIDが入った変数
 * @desc ロードするスロットのIDを指定する変数
 * @type variable
 *
 * @command loadLastSlot
 * @text 【ロード】最後にセーブしたスロットからロード
 * @desc 最後にセーブしたスロットからロードします
 */

(function () {
    'use strict';

    const Torigoya = (window.Torigoya = window.Torigoya || {});

    function getPluginName() {
        const cs = document.currentScript;
        return cs ? cs.src.split('/').pop().replace(/\.js$/, '') : 'TorigoyaMZ_LoadCommand';
    }

    function readParameter() {
        const parameter = PluginManager.parameters(getPluginName());
        return {
            version: '1.0.1',
        };
    }

    Torigoya.LoadCommand = {
        name: getPluginName(),
        parameter: readParameter(),
        isBusy: false,
        requestCall: undefined,
    };

    (() => {
        // -------------------------------------------------------------------------
        // Game_Interpreter

        /**
         * セーブファイルが存在するかチェック
         * @param {number} savefileId
         * @returns {boolean}
         */
        function doesSavefileExist(savefileId) {
            return DataManager.savefileExists(savefileId);
        }

        /**
         * ロードの実行リクエスト
         * @param {number} savefileId
         * @returns {Promise<*>}
         */
        Game_Interpreter.prototype.torigoyaLoadCommand_requestLoad = function (savefileId) {
            // セーブファイルの存在チェック
            if (!doesSavefileExist(savefileId)) {
                console.warn(`Save file ${savefileId} does not exist.`);
                return;
            }

            Torigoya.LoadCommand.requestCall = () => {
                return DataManager.loadGame(savefileId).then(() => {
                    $gamePlayer.reserveTransfer(
                        $gameMap.mapId(),
                        $gamePlayer.x,
                        $gamePlayer.y,
                        $gamePlayer.direction(),
                        0
                    );
                    $gameSystem.onAfterLoad();
                });
            };
        };

        Game_Interpreter.prototype.torigoyaLoadCommand_clearState = function () {
            Torigoya.LoadCommand.isBusy = false;
            Torigoya.LoadCommand.requestCall = undefined;
        };

        const upstream_Game_Interpreter_updateWait = Game_Interpreter.prototype.updateWait;
        Game_Interpreter.prototype.updateWait = function () {
            if (Torigoya.LoadCommand.requestCall) {
                const requestCall = Torigoya.LoadCommand.requestCall;
                requestCall()
                    .then(() => this.torigoyaLoadCommand_clearState())
                    .catch(() => this.torigoyaLoadCommand_clearState());

                Torigoya.LoadCommand.requestCall = undefined;
                Torigoya.LoadCommand.isBusy = true;
                return true;
            }
            if (Torigoya.LoadCommand.isBusy) return true;

            return upstream_Game_Interpreter_updateWait.apply(this);
        };

        // -------------------------------------------------------------------------
        // プラグインコマンド

        function commandLoadWithId({ slotId }) {
            slotId = parseInt(slotId, 10);
            this.torigoyaLoadCommand_requestLoad(slotId);
        }

        function commandLoadWithVariable({ slotVariableId }) {
            slotVariableId = parseInt(slotVariableId, 10);
            const slotId = $gameVariables.value(slotVariableId);
            this.torigoyaLoadCommand_requestLoad(slotId);
        }

        function commandLoadLastSlot() {
            const slotId = $gameSystem.savefileId() || DataManager.emptySavefileId();
            this.torigoyaLoadCommand_requestLoad(slotId);
        }

        PluginManager.registerCommand(Torigoya.LoadCommand.name, 'loadWithId', commandLoadWithId);
        PluginManager.registerCommand(Torigoya.LoadCommand.name, 'loadWithVariable', commandLoadWithVariable);
        PluginManager.registerCommand(Torigoya.LoadCommand.name, 'loadLastSlot', commandLoadLastSlot);
    })();
})();